/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - Xmlnet netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t exmlnet_net;

static int xmlnet_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "xmlnet") == 0)
		return 100;
	return 0;
}


/* print a key="val" pair making sure val is in valid xml syntax */
static void fprint_xmlprop(FILE *f, const char *akey, const char *aval)
{
	const char *s;

	fprintf(f, "%s=\"", akey);
	for(s = aval; *s != '\0'; s++) {
		if (*s == '"')
			fprintf(f, "&quot;");
		else if (*s == '&')
			fprintf(f, "&amp;");
		else if (*s < 32)
			fprintf(f, "&#x%x;", *s);
		else
			fputc(*s, f);
	}
	fputc('"', f);
}

/* print text, valid for xml */
static void fprint_xmltext(FILE *f, const char *txt)
{
	const char *s;

	for(s = txt; *s != '\0'; s++) {
		if (*s == '&')
			fprintf(f, "&amp;");
		else if (*s == '<')
			fprintf(f, "&lt;");
		else if (*s == '>')
			fprintf(f, "&gt;");
		else if (*s < 32)
			fprintf(f, "&#x%x;", *s);
		else
			fputc(*s, f);
	}
}

/* Write a single object attribute to f */
static void export_tag_key_val(FILE *f, const char *ind, const char *tag, const char *key, const char *val)
{
	if ((key == NULL) || (val == NULL))
		return;
	fprintf(f, "%s<%s ", ind, tag);
	fprint_xmlprop(f, "Key", key);
	fprint_xmlprop(f, "Value", val);
	fprintf(f, "/>\n");
}

/* Write a single <tag>val</tag> to f */
static void export_tag_val(FILE *f, const char *ind, const char *tag, const char *val)
{
	if (val == NULL) {
		fprintf(f, "%s<%s/>\n", ind, tag);
		return;
	}

	fprintf(f, "%s<%s>", ind, tag);
	fprint_xmltext(f, val);
	fprintf(f, "</%s>\n", tag);
}


/* Export comptag and nettag for attributes that have export_name; the user
   wanted these to end up on the netlist as custom tags/attributes/properties */
static void export_custom_attrs(FILE *f, const csch_ahdr_t *obj)
{
	htsp_entry_t *e;
	for(e = htsp_first(&obj->attr); e != NULL; e = htsp_next(&obj->attr, e)) {
		csch_attrib_t *a = e->value;
		if ((a->export_name != NULL) && (a->val != NULL))
			export_tag_key_val(f, "   ", "Attribute", a->export_name, a->val);
	}
}

/* Export abstract components: important fields and attributes */
static void xmlnet_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e, *p;


	fprintf(f, " <Components>\n");

	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp, *dev, *val;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);
		val = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE);
		dev = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE);

		fprintf(f, "  <Component ");
		fprint_xmlprop(f, "Refdes", refdes);
		fprintf(f, ">\n");
		export_tag_val(f, "   ", "Footprint", fp);
		export_tag_val(f, "   ", "Value", val);
		export_tag_val(f, "   ", "Device", dev);
		export_custom_attrs(f, &comp->hdr);


		/* Print terminals so that terminal symbolic names can be preserved.
		   (The netlist uses pin number to reference the terminal, symbolic name
		   is just an useful hint) */
		fprintf(f, "   <Pins>\n");
		for(p = htsp_first(&comp->ports); p != NULL; p = htsp_next(&comp->ports, p)) {
			const csch_aport_t *port = p->value;
			const char *pinnums = sch_nle_get_pinnum(port);
			const char *pinname = sch_nle_get_alt_attr(&port->hdr.attr, SCH_NLE_ALTATTR_PINNAME);

			if (pinname == NULL)
				continue;

			/* call export_attr on each pin in pinnums (which can be a list of
			   pin numbers) so that each pinnum -> pinname is written */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					fprintf(f, "    <Pin ");
					fprint_xmlprop(f, "Number", my_num);
					fprintf(f, ">\n");
					export_tag_val(f, "     ", "PinName", pinname);
					export_custom_attrs(f, &port->hdr);
					fprintf(f, "    </Pin>\n");
				}
			);
		}
		fprintf(f, "   </Pins>\n");
		fprintf(f, "  </Component>\n");
	}

	fprintf(f, " </Components>\n");
}

/* Export abstract nets; should exports net's name and a list of
   refdes-pinnum pairs connected and attributes */
static void xmlnet_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	fprintf(f, " <Nets>\n");

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);


		if (net->hdr.omit || (net->conns.used == 0)) continue;

		fprintf(f, "  <Net ");
		fprint_xmlprop(f, "Name", netname);
		fprintf(f, ">\n");

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "xmlnet: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "xmlnet: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "xmlnet: can't determine port's parent component refdes\n");*/
				continue;
			}

			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					
					fprintf(f, "   <Connection ");
					fprint_xmlprop(f, "Refdes", refdes);
					fputc(' ', f);
					fprint_xmlprop(f, "PinNum", my_num);
					fprintf(f, "/>\n");
				}
			);

		}
		export_custom_attrs(f, &net->hdr);
		fprintf(f, "  </Net>\n");
	}

	fprintf(f, " </Nets>\n");
}


/* Export netlist from the abstract model */
static int xmlnet_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "<?xml version='1.0' encoding='US-ASCII'?>\n");
	fprintf(f, "<Netlist BySoftware=\"sch-rnd\" ByModule=\"export_xmlnet\">\n");

	xmlnet_export_comps(f, abs);
	xmlnet_export_nets(f, abs);

	fprintf(f, "</Netlist>\n");
	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_xmlnet(int ver_needed) { return 0; }

void pplg_uninit_export_xmlnet(void)
{
	csch_plug_io_unregister(&exmlnet_net);
	rnd_export_remove_opts_by_cookie(xmlnet_cookie);
	rnd_hid_remove_hid(&xmlnet_hid);
}

int pplg_init_export_xmlnet(void)
{
	RND_API_CHK_VER;

	exmlnet_net.name = "export netlist to generic xml";
	exmlnet_net.export_prio = xmlnet_export_prio;
	exmlnet_net.export_project_abst = xmlnet_export_project_abst;
	exmlnet_net.ext_export_project = ".xml";
	csch_plug_io_register(&exmlnet_net);


	rnd_hid_nogui_init(&xmlnet_hid);

	xmlnet_hid.struct_size = sizeof(rnd_hid_t);
	xmlnet_hid.name = "xmlnet";
	xmlnet_hid.description = "Exports project's generic xml netlist";
	xmlnet_hid.exporter = 1;

	xmlnet_hid.get_export_options = xmlnet_get_export_options;
	xmlnet_hid.do_export = xmlnet_do_export;
	xmlnet_hid.parse_arguments = xmlnet_parse_arguments;
	xmlnet_hid.argument_array = xmlnet_values;

	xmlnet_hid.usage = xmlnet_usage;

	rnd_hid_register_hid(&xmlnet_hid);
	rnd_hid_load_defaults(&xmlnet_hid, xmlnet_options, NUM_OPTIONS);


	return 0;
}

